from pages.base_page import BasePage
from selenium.webdriver.support.wait import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC


class FormPage(BasePage):
    PAGE_URL = "https://practice-automation.com/form-fields/"

    NAME_INPUT = ("id", "name-input")
    PASSWORD_INPUT = ("xpath", "//input[@type='password']")

    DRINK_WATER = ("id", "drink1")
    DRINK_MILK = ("id", "drink2")
    DRINK_COFFEE = ("id", "drink3")
    DRINK_WINE = ("id", "drink4")
    DRINK_CTRL_ALT_DELIGHT = ("id", "drink5")

    COLOR_RED = ("id", "color1")
    COLOR_BLUE = ("id", "color2")
    COLOR_YELLOW = ("id", "color3")
    COLOR_GREEN = ("id", "color4")
    COLOR_PINK = ("id", "color5")

    AUTOMATION_SELECT = ("id", "automation")
    AUTOMATION_TOOLS_LIST = ("xpath", "//label[text()='Automation tools']/following-sibling::ul/li")
    EMAIL_INPUT = ("id", "email")
    MESSAGE_TEXTAREA = ("id", "message")

    SUBMIT_BUTTON = ("id", "submit-btn")

    def enter_name(self, name):
        self.driver.find_element(*self.NAME_INPUT).send_keys(name)

    def enter_password(self, password):
        self.driver.find_element(*self.PASSWORD_INPUT).send_keys(password)

    def enter_message(self, message):
        textarea = self.driver.find_element(*self.MESSAGE_TEXTAREA)
        textarea.clear()
        textarea.send_keys(message)

    def select_drink(self, drink_name):
        drink_locators = {
            "Water": self.DRINK_WATER,
            "Milk": self.DRINK_MILK,
            "Coffee": self.DRINK_COFFEE,
            "Wine": self.DRINK_WINE,
            "Ctrl-Alt-Delight": self.DRINK_CTRL_ALT_DELIGHT
        }
        if drink_name in drink_locators:
            element = self.driver.find_element(*drink_locators[drink_name])
            self.driver.execute_script("arguments[0].scrollIntoView({block: 'center', behavior: 'instant'});", element)
            element.click()

    def select_color(self, color_name):
        color_locators = {
            "Red": self.COLOR_RED,
            "Blue": self.COLOR_BLUE,
            "Yellow": self.COLOR_YELLOW,
            "Green": self.COLOR_GREEN,
            "#FFC0CB": self.COLOR_PINK
        }
        if color_name in color_locators:
            self.find_and_click(color_locators[color_name])

    def select_automation_opinion(self, opinion):
        select_element = self.driver.find_element(*self.AUTOMATION_SELECT)
        from selenium.webdriver.support.ui import Select
        select = Select(select_element)
        select.select_by_value(opinion)

    def enter_email(self, email):
        self.driver.find_element(*self.EMAIL_INPUT).send_keys(email)

    def get_automation_tools_list(self):
        tools_elements = self.driver.find_elements(*self.AUTOMATION_TOOLS_LIST)
        return [tool.text for tool in tools_elements]

    def fill_message_with_automation_tools(self):
        tools_list = self.get_automation_tools_list()
        tools_text = "\n".join(tools_list)
        self.enter_message(tools_text)
        return tools_text

    def click_submit(self):
        self.find_and_click(self.SUBMIT_BUTTON)

    def get_alert_text(self):
        WebDriverWait(self.driver, 2).until(EC.alert_is_present())
        alert = self.driver.switch_to.alert
        text = alert.text
        alert.accept()
        return text

    def is_name_field_valid(self):
        name_input = self.driver.find_element(*self.NAME_INPUT)
        return name_input.get_attribute("value") != ""
